<?php
/**
 * Plugin Name: CapitalPay Gateway for WooCommerce
 * Description: Integrates CapitalPay acquiring service into WooCommerce checkout.
 * Version: 1.0.0
 * Author: Collect Group team
 */

if (!defined('ABSPATH')) exit;

add_action('plugins_loaded', 'init_capitalpay_gateway');

function init_capitalpay_gateway()
{
    if (!class_exists('WC_Payment_Gateway')) return;

    class WC_Gateway_CapitalPay extends WC_Payment_Gateway
    {

        public function __construct()
        {

            $this->id = 'capitalpay';
            $this->method_title = 'CapitalPay';
            $this->method_description = 'Payment via CapitalPay by VISA or MASTER CARD';
            $this->has_fields = false;
            $this->supports = [
                'products',
                'virtual',
                'woocommerce_blocks',
                'tokenization',
                'add_payment_method'
            ];

            $this->init_form_fields();
            $this->init_settings();

            $this->title = $this->get_option('title');
            $this->description = $this->get_option('description');
            $this->cash_desk_uuid = $this->get_option('cash_desk_uuid');
            $this->secret_key = $this->get_option('secret_key');
            $this->enabled = $this->get_option('enabled');
            $this->language = $this->get_option('language');

            add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);
            add_action('woocommerce_blocks_loaded', [$this, 'register_block_support']);
        }

        public function register_block_support()
        {
            if (!class_exists('Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType')) {
                return;
            }

            require_once plugin_dir_path(__FILE__) . 'class-wc-capitalpay-blocks-support.php';
            add_action(
                'woocommerce_blocks_payment_method_type_registration',
                function (Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry $payment_method_registry) {
                    $payment_method_registry->register(new WC_CapitalPay_Blocks_Support());
                }
            );
        }

        public function payment_fields()
        {
            if ($this->description) {
                echo wpautop(wp_kses_post($this->description));
            }
        }

        public function is_active()
        {
            $is_active = $this->gateway->enabled === 'yes';

            if ($is_active && function_exists('WC') && WC()->cart) {
                $is_active = WC()->cart->needs_payment();
            }

            error_log('CapitalPay is_active check: ' . ($is_active ? 'true' : 'false'));
            return $is_active;
        }

        public function init_form_fields()
        {
            $this->form_fields = [
                'enabled' => [
                    'title' => 'Enable',
                    'type' => 'checkbox',
                    'label' => 'Enable CapitalPay',
                    'default' => 'yes'
                ],
                'title' => [
                    'title' => 'Method name',
                    'type' => 'text',
                    'default' => 'CapitalPay'
                ],
                'description' => [
                    'title' => 'Description',
                    'type' => 'textarea',
                    'description' => 'Description of the payment method for the user.',
                    'default' => 'Payment via CapitalPay by VISA or MASTER CARD',
                ],

                'language' => [
                    'title'   => 'Language',
                    'type'    => 'select',
                    'options' => [
                        'en' => 'English',
                        'ru' => 'Русский',
                    ],
                    'default' => 'en',
                    'description' => 'Choose the language for the payment page.',
                ],

                'cash_desk_uuid' => [
                    'title' => 'Cash Desk UUID',
                    'type' => 'text'
                ],
                'secret_key' => [
                    'title' => 'Secret Key',
                    'type' => 'password'
                ]
            ];
        }

        public function process_payment($order_id)
        {
            $order = wc_get_order($order_id);
            $amount = $order->get_total();
            $currency = $order->get_currency();
            $external_data = json_encode(["email" => $order->get_billing_email()]);
            $external_order_id = $order->get_id();
            $external_client_id = $order->get_customer_id();
            if (empty($external_client_id) || !is_numeric($external_client_id)) {
                $external_client_id = '0';
            } else {
                $external_client_id = (string)$external_client_id;
            }


            $fields = [
                "amount" => $amount,
                "currency_code" => $currency,
                "external_client_id" => $external_client_id,
                "external_data" => $external_data,
                "external_order_id" => $external_order_id,
            ];
            $fields_order = ["amount", "currency_code", "external_client_id", "external_data", "external_order_id"];

            $signature = generate_signature($fields, $fields_order, $this->secret_key);

            $request_data = [
                "amount" => $amount,
                "currency_code" => $currency,
                "cash_desk_uuid" => $this->cash_desk_uuid,
                "title" => "Order #$external_order_id",
                "description" => "Payment for the order #$external_order_id",
                "external_client_id" => $external_client_id,
                "external_data" => $external_data,
                "external_order_id" => $external_order_id,
                "locale" => $this->language,
                "fail_url" => $order->get_cancel_order_url(),
                "success_url" => $this->get_return_url($order),
                "webhook_url" => home_url('/?wc-api=capitalpay-webhook'),
            ];

            $response = wp_remote_post('https://api.capitalpay.biz/api/public/order', [
                'method' => 'POST',
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                    'Signature' => $signature
                ],
                'body' => json_encode($request_data)
            ]);

            if (is_wp_error($response)) {
                wc_add_notice('Error connecting to payment gateway.', 'error');
                return false;
            }

            $body = json_decode(wp_remote_retrieve_body($response), true);

            if (!empty($body['data']['url'])) {
                return [
                    'result' => 'success',
                    'redirect' => $body['data']['url']
                ];
            } else {
                wc_add_notice('Error creating payment.', 'error');
                return false;
            }
        }
    }

    add_action('wp_enqueue_scripts', function () {
        if (function_exists('is_checkout') && is_checkout()) {
            wp_enqueue_script('wc-capitalpay-blocks');
        }
    }, 20);

    class WC_CapitalPay_Blocks_Support extends Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType
    {
        protected $gateway;
        protected $settings;

        public function __construct()
        {
            $this->name = 'capitalpay';
            $this->gateway = new WC_Gateway_CapitalPay();
        }

        public function initialize()
        {
            $this->settings = $this->gateway->settings;
        }

        public function is_active()
        {
            return $this->gateway->is_available();
        }

        public function get_payment_method_script_handles()
        {
            $script_path = plugin_dir_path(__FILE__) . 'assets/js/blocks.js';

            if (file_exists($script_path)) {
                $dependencies = [
                    'react',
                    'react-dom',
                    'wc-blocks-registry',
                    'wp-element',
                    'wp-i18n',
                    'wp-components'
                ];

                wp_register_script(
                    'wc-capitalpay-blocks',
                    plugins_url('assets/js/blocks.js', __FILE__),
                    ['react', 'react-dom', 'wp-element', 'wc-blocks-registry'],
                    filemtime($script_path),
                    true
                );

                wp_localize_script(
                    'wc-capitalpay-blocks',
                    'wc_capitalpay_params',
                    [
                        'title' => $this->gateway->title,
                        'description' => $this->gateway->description,
                        'supports' => $this->gateway->supports
                    ]
                );

                return ['wc-capitalpay-blocks'];
            }
            return [];
        }

        public function get_payment_method_data()
        {
            return [
                'title' => $this->gateway->title,
                'description' => $this->gateway->description,
                'supports' => (array)$this->gateway->supports,
                'icon' => ''
            ];
        }
    }

    function add_capitalpay_gateway($methods)
    {
        $methods[] = 'WC_Gateway_CapitalPay';
        return $methods;
    }

    add_filter('woocommerce_payment_gateways', 'add_capitalpay_gateway');

    add_filter('woocommerce_blocks_payment_method_script_handles', function ($script_handles, $payment_method_name) {
        if ($payment_method_name === 'capitalpay') {
            $script_handles[] = 'wc-capitalpay-blocks';
        }
        return $script_handles;
    }, 10, 2);

    add_action('woocommerce_blocks_loaded', function () {
        if (class_exists('Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry')) {
            $container = Automattic\WooCommerce\Blocks\Package::container();
            $registry = $container->get(Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry::class);
            $registry->register(new WC_CapitalPay_Blocks_Support());
        } else {
            error_log('WooCommerce Blocks PaymentMethodRegistry not found');
        }
    }, 20);

    add_action('plugins_loaded', 'load_capitalpay_textdomain');

    add_action('woocommerce_api_capitalpay-webhook', 'handle_capitalpay_webhook');

    function handle_capitalpay_webhook()
    {
        $log_prefix = '[CapitalPay Webhook] ';

        $payload = file_get_contents('php://input');

        $data = json_decode($payload, true)['data'];
        if (empty($data)) {
            error_log($log_prefix . 'Invalid or empty JSON payload.');
            status_header(400);
            echo 'Invalid data';
            exit;
        }

        $headers = function_exists('getallheaders') ? getallheaders() : [];
        $signatureFromRequest = isset($headers['Signature']) ? $headers['Signature'] : null;

        if (!$signatureFromRequest) {
            error_log($log_prefix . 'Missing Signature header.');
            status_header(400);
            echo 'Missing Signature';
            exit;
        }

        $order_id = isset($data['external_order_id']) ? intval($data['external_order_id']) : 0;
        $status = isset($data['status']) ? sanitize_text_field($data['status']) : '';

        if (!$order_id || !$status) {
            error_log($log_prefix . 'Missing order_id or status in payload.');
            status_header(400);
            echo 'Missing order ID or status';
            exit;
        }

        $order = wc_get_order($order_id);
        if (!$order) {
            error_log($log_prefix . "Order not found: $order_id");
            status_header(404);
            echo 'Order not found';
            exit;
        }

        $fields = [
            "amount" => $data['amount'],
            "currency_code" => $data['currency_code'],
            "error_code" => $data['error_code'],
            "error_message" => $data['error_message'],
            "external_client_id" => $data['external_client_id'],
            "external_order_id" => $data['external_order_id'],
            "order_uuid" => $data['order_uuid'],
            "status" => $data['status'],
            "status_id" => $data['status_id'],

        ];
        $fields_order = [
            "amount",
            "currency_code",
            "error_code",
            "error_message",
            "external_client_id",
            "external_order_id",
            "order_uuid",
            "status",
            "status_id"
        ];

        $gateway = new WC_Gateway_CapitalPay();

        $generated_signature = generate_signature($fields, $fields_order, $gateway->secret_key);

        if ($signatureFromRequest !== $generated_signature) {
            error_log($log_prefix . 'Signature mismatch.');
            error_log($log_prefix . 'Signature request: ' . $signatureFromRequest);
            error_log($log_prefix . 'Signature verify:  ' . $generated_signature);
            status_header(400);
            echo 'Invalid signature';
            exit;
        }

        error_log($log_prefix . "Processing order #$order_id with status: $status");

        switch ($status) {
            case 'Paid':
                $order->payment_complete();
                $order->add_order_note('Payment completed via CapitalPay webhook.');
                break;
            case 'Failed':
                $order->update_status('failed', 'Payment failed via CapitalPay webhook.');
                break;
            case 'Canceled':
                $order->update_status('cancelled', 'Payment cancelled via CapitalPay webhook.');
                break;
            default:
                $order->add_order_note('Received unknown payment status from CapitalPay: ' . $status);
                break;
        }

        error_log($log_prefix . "Order #$order_id updated successfully.");
        status_header(200);
        echo 'OK';
        exit;
    }

    function generate_signature($fields, $fields_order, $key)
    {
        $signatureDataStr = '';
        foreach ($fields_order as $k) {
            $signatureDataStr .= (isset($fields[$k]) ? $fields[$k] : '') . ";";
        }
        error_log('Signature generate:' . $signatureDataStr);
        error_log('Signature key:' . $key);
        return hash_hmac('sha256', $signatureDataStr, $key);
    }

}

